/*
 Program WinCaml: Graphical User Interface
 for interactive use of Caml-Light and Ocaml.
 Copyright (C) 2005-2018 Jean Mouric 35700 Rennes France
 email: jean.mouric@orange.fr
 
 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

// File Qt/Paths.cpp

#include "Qt.h"

#ifndef Q_OS_WIN
#include <stdio.h>
#include <sys/stat.h>
#include <iostream>
#include <fstream>

string fromUnicode(const wstring& wstr, int code);
wstring toUnicode(const string& str, int code);
string toBytes(const wstring&);
wstring setCamlPath(const wstring&);
wstring getCamlPath(const wstring&);
void ldconf(const wstring& ocamlPath);
bool resetCamlPath(const wstring&);
bool createDirectory(const wstring& dirPath);

extern bool camlnew;
extern CMDIFrame* mainFrame;

QString applicationDirPath()
{
#ifdef Q_OS_MAC
    QDir qd(qApp->applicationDirPath());
    qd.cdUp(); qd.cdUp(); qd.cdUp();
    return qd.path();
#else
    return qApp->applicationDirPath();
#endif
}
bool getConfigFolderPath(wstring& path)
{
    path = (applicationDirPath() + "/." + QDir(QDir::homePath()).dirName() + "/").toStdWString();
    return true;
}
string toUTF8(const wstring& wstr)
{
    return fromUnicode(wstr, ENCODING_UTF8);
}
void ldconf(const wstring& ocamllib)
{
    wstring ldconfPath = ocamllib + L"/ld.conf";
    fstream fs(toBytes(ldconfPath).c_str(), fstream::out);
    if (!fs.fail())
    {
        fs << toUTF8(ocamllib + L"/stublibs").c_str() << endl;
        fs << toUTF8(ocamllib).c_str() << endl;
        fs.close();
    }
}
wstring fromUTF8(const string& str)
{
    return toUnicode(str, ENCODING_UTF8);
}
wstring getCamlPath(const wstring& camlbase)
{
    wstring cfp;
    string camlPath;
    if (getConfigFolderPath(cfp))
    {
        cfp += camlbase;
        fstream fs(toBytes(cfp).c_str(), fstream::in);
        if (!fs.fail())
        {
            char c;
            while (fs.get(c) && c != L'\r' && c != L'\n')
                camlPath += c;
            fs.close();
        }
    }
    return fromUTF8(camlPath);
}
void writeCamlPath(QString camlbase, QString camlPath)
{
    wstring s;
    if (getConfigFolderPath(s) && createDirectory(s))
    {
        s += camlbase.toStdWString();
        fstream fs(toBytes(s).c_str(), fstream::out);
        if (!fs.fail())
        {
            fs << toUTF8(camlPath.toStdWString()) << endl;
            fs.close();
        }
    }
}
QString setCamlPath(bool ocaml)
{
    wstring distrib = ocaml ? L"OCaml" : L"Caml Light";
    wstring message = L"Le dialogue suivant devrait permettre de naviguer vers un dossier " + distrib + L" valide ou d'annuler.";
    ::infoMessage(message);
    QString prompt = ocaml ? "Emplacement d'OCaml ?" : "Emplacement de Caml Light ?";
    return QFileDialog::getExistingDirectory(mainFrame, prompt, "", QFileDialog::ShowDirsOnly | QFileDialog::ReadOnly);
}

bool resetCamlPath(const QString& camlbase)
{
    wstring s;
    if (getConfigFolderPath(s))
    {
        void deleteFile(const wstring& fileName);
        s += camlbase.toStdWString();
        QFileInfo fi = QString::fromStdWString(s);
        if (fi.exists())
        {
            QString distrib = camlbase == "camlbase.txt" ? "Caml Light" : "OCaml";
            QString message = "Abandonner la distribution " + distrib + " actuelle ?";
            if (::yesnoMessage(message.toStdWString()))
            {
                deleteFile(s);
            }
            return true;
        }
    }
    return false;
}

bool resetCamlPaths()
{
    bool b1 = resetCamlPath("camlbase.txt");
    bool b2 = resetCamlPath("ocamlbase.txt");
    return b1 || b2;
}

#ifdef Q_OS_LINUX
char byte64(const char* path)
{
    unsigned char c[5] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return 0;
    if (fread(c, 5, 1, f) == 1)
    {
        fclose(f);
        return c[4];
    }
    return 0;
}
#endif

bool isExecutable(const QString& runtimePath)
{
    FILE* f = fopen(runtimePath.toStdString().c_str(), "rb");
    if (!f) return false;
    char c[5];
    if (fread(c, 5, 1, f) == 1)
    {
        fclose(f);
#ifdef Q_OS_LINUX
        if (c[0] != 0x7f || c[1] != 0x45 || c[2] != 0x4c || c[3] != 0x46 || c[4] != byte64("/bin/sh"))
#else
            uint32_t m = *(uint32_t*)c;
        if (m != 0xfeedface && m != 0xfeedfacf && m != 0xcefaedfe && m != 0xcffaedfe)
#endif
        {
            errorMessage(L"Format de fichier binaire incorrect");
            return true;
        }
    }
    return true;
}

#ifdef Q_OS_LINUX
bool isElf64leExecutable(const char* path)
{
    unsigned char c[6] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return false;
    if (fread(c, 6, 1, f) == 1)
    {
        fclose(f);
        return c[0] == 0x7f && c[1] == 0x45 && c[2] == 0x4c && c[3] == 0x46 && c[4] == 0x02 && c[5] == 0x01;
    }
    return false;
}
#else
bool isMac64Executable(const char* path)
{
    unsigned char c[4] = {0};
    FILE* f = fopen(path, "rb");
    if (!f) return false;
    if (fread(c, 4, 1, f) == 1)
    {
        fclose(f);
        return c[0] == 0xcf && c[1] == 0xfa && c[2] == 0xed && c[3] == 0xfe;
    }
    return false;
}
#endif

bool findOCaml(QString& camlPath)
{
    QFileInfo fi1 = QString("/usr/local/bin/ocamlrun");
    QFileInfo fi2 = QString("/usr/local/bin/ocaml");
    if (!fi1.exists() || !fi2.exists() || !fi1.isExecutable() || !isExecutable("/usr/local/bin/ocamlrun"))
    {
        camlPath = "/usr";
        fi1 = QString("/usr/bin/ocamlrun");
        fi2 = QString("/usr/bin/ocaml");
        if (!fi1.exists() || !fi2.exists() || !fi1.isExecutable() || !isExecutable("/usr/bin/ocamlrun"))
        {
            camlPath = QString::fromStdWString(getCamlPath(L"ocamlbase.txt"));
            fi1 = camlPath + "/bin/ocamlrun";
            fi2 = camlPath + "/bin/ocaml";
            while (!fi1.exists() || !fi2.exists() || !fi1.isExecutable() || !isExecutable(camlPath + "/bin/ocamlrun"))
            {
                QString path = setCamlPath(true);
                if (path.isEmpty() || path.isNull()) return false;
                camlPath = path;
                fi1 = camlPath + "/bin/ocamlrun";
                fi2 = camlPath + "/bin/ocaml";
            }
        }
    }
    return true;
}

bool findCamllight(QString& camlPath, QString& toplevel, QString& camlLib)
{
    QFileInfo fi = toplevel;
    if (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
    {
        
        camlPath = "/usr/local";
        camlLib = camlPath + "/lib/caml-light";
        toplevel = camlLib + "/caml_all";
        fi = toplevel;
        if (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
        {
            camlPath = "/usr";
            camlLib = camlPath + "/lib/caml-light";
            toplevel = camlLib + "/caml_all";
            fi = toplevel;
            if (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
            {
                camlPath = QString::fromStdWString(getCamlPath(L"camlbase.txt"));
                camlLib = camlPath + "/lib/caml-light";
                toplevel = camlLib + "/caml_all";
                fi = toplevel;
                while (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
                {
                    QString path = setCamlPath(false);
                    if (path.isEmpty() || path.isNull()) return false;
                    camlPath = path;
                    camlLib = camlPath + "/lib/caml-light";
                    toplevel = camlLib + "/caml_all";
                    fi = toplevel;
                }
            }
        }
        
    }
    return true;
}

bool findOCaml1(QString& camlPath)
{
    QString path = setCamlPath(true);
    if (path.isEmpty() || path.isNull()) {camlPath = ""; return false;}
    QFileInfo fi1 = path + "/bin/ocamlrun";
    QFileInfo fi2 = path + "/bin/ocaml";
    while (!fi1.exists() || !fi2.exists() || !fi1.isExecutable() || !isExecutable(path + "/bin/ocamlrun"))
    {
        path = setCamlPath(true);
        if (path.isEmpty() || path.isNull()) {camlPath = ""; return false;}
        fi1 = camlPath + "/bin/ocamlrun";
        fi2 = camlPath + "/bin/ocaml";
    }
    camlPath = path;
    return true;
}

bool findCamllight1(QString& camlPath, QString& toplevel, QString& camlLib)
{
    camlPath = setCamlPath(false);
    if (camlPath.isEmpty() || camlPath.isNull()) {camlPath = ""; return false;}
    camlLib = camlPath + "/lib/caml-light";
    toplevel = camlLib + "/caml_all";
    QFileInfo fi = toplevel;
    while (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
    {
        camlPath = setCamlPath(false);
        if (camlPath.isEmpty() || camlPath.isNull())  {camlPath = ""; return false;}
        camlLib = camlPath + "/lib/caml-light";
        toplevel = camlLib + "/caml_all";
        fi = toplevel;
    }
    return true;
}

CProcess* startCaml1(bool ocaml)
{
    //#ifdef Q_OS_MAC
    QDir::setCurrent(applicationDirPath());
    //#endif
    if (ocaml)
    {
        QString ocamlPath = "/usr/local";
        if (!findOCaml(ocamlPath)) return NULL;
        if (ocamlPath != "/usr/local" && ocamlPath != "/usr") writeCamlPath("ocamlbase.txt", ocamlPath);
        
#ifdef Q_OS_LINUX
        QString graphlib = isElf64leExecutable((ocamlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/linux/ocaml64" : qApp->applicationDirPath() + "/ocaml/linux/ocaml32";
#else
        QString graphlib = isMac64Executable((ocamlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/mac/ocaml64" : qApp->applicationDirPath() + "/ocaml/mac/ocaml32";
#endif
        
        QString ocamllib = ocamlPath + "/lib/ocaml";
        if (!QDir(ocamllib).exists()) ocamllib = ocamlPath + "/lib";
        ldconf(ocamllib.toStdWString());
        
        QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
        env.insert("OCAMLLIB", ocamllib);
        env.insert("CAML_LD_LIBRARY_PATH", ocamllib + "/stublibs");
        env.insert("PATH", ocamlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
        
        QStringList args = (QStringList(ocamlPath + "/bin/ocaml") << "-I" << graphlib);
        
        CProcess* process = new CProcess();
        process->setProcessEnvironment(env);
        process->start(ocamlPath + "/bin/ocamlrun", args);
        return process;
    }
    else
    {
        QString camlPath = applicationDirPath() + "/caml-light";
        QString path = camlPath;
        QString camlLib = camlPath + "/lib/caml-light";
#ifdef Q_OS_LINUX
        QString toplevel = isElf64leExecutable("/bin/sh") ? camlPath + "/bin/lintoplevel64" : camlPath + "/bin/lintoplevel32";
#else
        QString toplevel = isMac64Executable("/bin/sh") ? camlPath + "/bin/mactoplevel64" : camlPath + "/bin/mactoplevel32";
#endif
        
        if (!findCamllight(camlPath, toplevel, camlLib)) return NULL;
        
        if (camlPath != "/usr/local" && camlPath != "/usr" && camlPath != path)
        {
            writeCamlPath("camlbase.txt", camlPath);
        }
        QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
        env.insert("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin:");
        env.insert("CAMLLIGHTDIR", camlPath);
        
        QString locale = QLocale::system().name().mid(0, 2);
        QStringList args = (QStringList() << "-stdlib" << camlLib << "-lang" << locale);
        
        CProcess* process = new CProcess();
        process->setProcessEnvironment(env);
        process->start(toplevel, args);
        return process;
    }
}

CProcess* startCaml(bool ocaml)
{
    wstring message = L"Changer de distribution ";
    message += (ocaml ? L"OCaml ?" : L"Caml Light ?");
    QDir::setCurrent(applicationDirPath());
    if (camlnew && ::yesnoMessage(message))
    {
        camlnew = false;
        if (ocaml)
        {
            QString camlPath;
            if (findOCaml1(camlPath))
            {
#ifdef Q_OS_LINUX
                QString graphlib = isElf64leExecutable((camlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/linux/ocaml64" : qApp->applicationDirPath() + "/ocaml/linux/ocaml32";
#else
                QString graphlib = isMac64Executable((camlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/mac/ocaml64" : qApp->applicationDirPath() + "/ocaml/mac/ocaml32";
#endif
                QString ocamllib = camlPath + "/lib/ocaml";
                if (!QDir(ocamllib).exists()) ocamllib = camlPath + "/lib";
                
                writeCamlPath("ocamlbase.txt", camlPath);
                ldconf(ocamllib.toStdWString());
                
                QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
                
                env.insert("OCAMLLIB", ocamllib);
                env.insert("CAML_LD_LIBRARY_PATH", ocamllib + "/stublibs");
                env.insert("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
                
                //QStringList args = (QStringList(camlPath + "/bin/ocaml"));
                QStringList args = (QStringList(camlPath + "/bin/ocaml") << "-I" << graphlib);
                CProcess* process = new CProcess();
                process->setProcessEnvironment(env);
                process->start(camlPath + "/bin/ocamlrun", args);
                return process;
            }
            else
            {
                if (getCamlPath(L"ocamlbase.txt") != L"" && ::yesnoMessage(L"abandonner la distribution OCaml actuelle ?"))
                {
                    resetCamlPath("ocamlbase.txt");
                    return startCaml1(ocaml);
                }
            }
        }
        else
        {
            QString camlPath;
            QString toplevel;
            QString camlLib;
            if (findCamllight1(camlPath, toplevel, camlLib))
            {
                writeCamlPath("camlbase.txt", camlPath);
                QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
                env.insert("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin:");
                env.insert("CAMLLIGHTDIR", camlPath);
                
                QString locale = QLocale::system().name().mid(0, 2);
                QStringList args = (QStringList() << "-stdlib" << camlLib << "-lang" << locale);
                
                CProcess* process = new CProcess();
                process->setProcessEnvironment(env);
                process->start(toplevel, args);
                return process;
            }
            else
            {
                if (getCamlPath(L"camlbase.txt") != L"" && ::yesnoMessage(L"abandonner la distribution Caml Light actuelle ?"))
                {
                    resetCamlPath("camlbase.txt");
                    return startCaml1(ocaml);
                }
            }
        }
    }
    camlnew = false;
    if (ocaml)
    {
        QString camlPath = QString::fromStdWString(getCamlPath(L"ocamlbase.txt"));
        QFileInfo fi1 = camlPath + "/bin/ocamlrun";
        QFileInfo fi2 = camlPath + "/bin/ocaml";
        if (!fi1.exists() || !fi2.exists() || !fi1.isExecutable() || !isExecutable(camlPath + "/bin/ocamlrun"))
        {
            return startCaml1(ocaml);
        }
        else
        {
            QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
#ifdef Q_OS_LINUX
            QString graphlib = isElf64leExecutable((camlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/linux/ocaml64" : qApp->applicationDirPath() + "/ocaml/linux/ocaml32";
#else
            QString graphlib = isMac64Executable((camlPath + "/bin/ocamlrun").toStdString().c_str()) ? applicationDirPath() + "/ocaml/mac/ocaml64" : qApp->applicationDirPath() + "/ocaml/mac/ocaml32";
#endif
            QString ocamllib = camlPath + "/lib/ocaml";
            if (!QDir(ocamllib).exists()) ocamllib = camlPath + "/lib";
            env.insert("OCAMLLIB", ocamllib);
            env.insert("CAML_LD_LIBRARY_PATH", ocamllib + "/stublibs");
            env.insert("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin");
            
            //QStringList args = (QStringList(camlPath + "/bin/ocaml"));
            QStringList args = (QStringList(camlPath + "/bin/ocaml") << "-I" << graphlib);
            CProcess* process = new CProcess();
            process->setProcessEnvironment(env);
            process->start(camlPath + "/bin/ocamlrun", args);
            return process;
            
        }
    }
    else
    {
        QString camlPath = QString::fromStdWString(getCamlPath(L"camlbase.txt"));
        QString camlLib = camlPath + "/lib/caml-light";
        QString toplevel = camlLib + "/caml_all";
        QFileInfo fi = toplevel;
        if (!fi.exists() || !fi.isExecutable() || !isExecutable(toplevel))
        {
            return startCaml1(ocaml);
        }
        else
        {
            QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
            env.insert("PATH", camlPath + "/bin:/usr/local/bin:/usr/bin:/bin:");
            env.insert("CAMLLIGHTDIR", camlPath);
            
            QString locale = QLocale::system().name().mid(0, 2);
            QStringList args = (QStringList() << "-stdlib" << camlLib << "-lang" << locale);
            
            CProcess* process = new CProcess();
            process->setProcessEnvironment(env);
            process->start(toplevel, args);
            return process;
        }
    }
}

void camllightHelp()
{
#ifdef Q_OS_MAC
    QStringList args = (QStringList() << "-a" << "safari.app" << applicationDirPath() + "/doc/man-caml/index.html");
    QProcess::startDetached("open", args);
#else
#ifdef Q_OS_LINUX
    QStringList args = (QStringList() << qApp->applicationDirPath() + "/doc/man-caml/index.html");
    QProcess::startDetached("/usr/bin/firefox", args);
#endif
#endif
}
void ocamlHelp()
{
#ifdef Q_OS_MAC
    QStringList args = (QStringList() << "-a" << "safari.app" << applicationDirPath() + "/doc/htmlman/index.html");
    QProcess::startDetached("open", args);
#else
#ifdef Q_OS_LINUX
    QStringList args = (QStringList() << qApp->applicationDirPath() + "/doc/htmlman/index.html");
    QProcess::startDetached("/usr/bin/firefox", args);
#endif
#endif
}
#endif

#ifdef Q_OS_WIN
#include "../Win/Paths.cpp"
#endif
